/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.ComponentModel;
using System.ComponentModel.Design;
using System.Globalization;
using System.Reflection;
using System.Drawing.Design; // UITypeEditor

using Borland.Eco.Exceptions;
using Borland.Eco.Services;
using Borland.Eco.UmlRt;
using Borland.Eco.Subscription;
using Borland.Eco.ObjectRepresentation;
using Borland.Eco.Globalization;

namespace Borland.Eco.Handles
{
	[Designer("Borland.Eco.Handles.Design.RootHandleDesigner, Borland.Eco.Handles.Design", typeof(IDesigner))]
	public abstract class RootHandle: ElementHandle, IHasTypeSystem
	{
		private sealed class ActiveChangedAdapter: SubscriberAdapterBase
		{
			protected override void DoReceive(object sender, EventArgs e, object actualSubscriber)
			{
				if (actualSubscriber == null) throw new ArgumentNullException("actualSubscriber"); // do not localize
				(actualSubscriber as RootHandle).ActiveChanged();
			}

			public ActiveChangedAdapter(object subscriber): base(subscriber) { }
		}

		private ActiveChangedAdapter m_ActiveChangedAdapter;
		protected RootHandle(): base()
		{
			m_ActiveChangedAdapter = new ActiveChangedAdapter(this);
		}

		protected sealed override IEcoTypeSystem GetTypeSystem()
		{
			ITypeSystemService tss = (ITypeSystemService)InternalGetEcoService(typeof(ITypeSystemService));
			if (tss == null)
				return null;
			else
				return tss.TypeSystem;
		}

		protected sealed override IOclTypeService GetOclTypeService()
		{
			return (IOclTypeService)InternalGetEcoService(typeof(IOclTypeService));
		}

		protected sealed override IOclPsTypeService GetOclPsTypeService()
		{
			return (IOclPsTypeService)InternalGetEcoService(typeof(IOclPsTypeService));
		}

		protected sealed override IActionLanguageTypeService GetActionLanguageTypeService()
		{
			return (IActionLanguageTypeService)InternalGetEcoService(typeof(IActionLanguageTypeService));
		}

		private void EcoSpaceChanged() // notify subclasses
		{
			CheckType();
			StaticContextChanged();
		}

		private EcoSpace m_EcoSpace;

		/// <summary>
		/// <para>The ECO Space the root handle is connected to.</para>
		/// <para>This property must be set programmatically at runtime.</para>
		/// </summary>
		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public EcoSpace EcoSpace
		{
			get { return m_EcoSpace; }
			set
			{
				if (EcoSpace != value)
				{
					EcoSpace oldOS = EcoSpace;
					m_EcoSpace = value;
					m_ActiveChangedAdapter.Deactivate();
					m_ActiveChangedAdapter = new ActiveChangedAdapter(this);
					if (EcoSpace != null)
						EcoSpace.SubscribeToActive(m_ActiveChangedAdapter);
					if ((oldOS == null) || (value == null))
						ActiveChanged();
					EcoSpaceChanged();
				}
			}
		}
#region IHasTypeSystem
		IEcoTypeSystem IHasTypeSystem.TypeSystem
		{
			get { return GetTypeSystem(); }
		}
		bool IHasTypeSystem.IsHookedUp { get { return EcoSpaceType != null; } }
#endregion

		private	string m_StaticValueTypeName = "";
		/// <summary>
		/// <para>This name determines the value of IStaticContext.StaticUmlType by
		/// looking it up in the typesystem for the Eco System.</para>
		///
		/// <para>This type is used at design time for support in the OCL editor,
		/// and at runtime for determining the available properties for the
		/// databinding if the Element is not set.</para>
		/// </summary>
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryTypes")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyStaticValueTypeName")]
		[Editor("Borland.Eco.Handles.Design.AnyTypeNameSelectorEditor, Borland.Eco.Handles.Design", typeof(UITypeEditor))]
		[DefaultValue("")]
		public string StaticValueTypeName
		{
			get { return m_StaticValueTypeName; }
			set
			{
				if (value == null)
					value = "";
				if (value != m_StaticValueTypeName)
				{

					m_StaticValueTypeName = value;
					StaticContextChanged();
				}
			}
		}

		protected override void Dispose(bool disposing)
		{
			try
			{
				if (disposing)
					EcoSpace = null;
			}
			finally
			{
				base.Dispose(disposing);
			}
		}

		protected sealed override IClassifier GetStaticUmlType()
		{
			if (GetTypeSystem() != null)
				return GetTypeSystem().GetClassifierByName(StaticValueTypeName);
			else
				return null;
		}
		protected sealed override object GetEcoService(System.Type serviceType)
		{
			if (Active)
				return InternalGetEcoService(serviceType);
			else
				return null;
		}

		private object InternalGetEcoService(System.Type serviceType)
		{
			if (EcoSpace != null)
				 return EcoSpace.GetEcoService(serviceType);

			System.Type osType = EcoSpaceType;
			if (osType == null)
				return null;
			else
			{
				MethodInfo mi = osType.GetMethod(DefaultEcoSpace.GetTypeSystemServiceName, BindingFlags.Static | BindingFlags.Public);
				if (mi == null)
					throw new EcoException(HandlesStringRes.sGetTypeSystemServiceMissing(this.ToString()));
				ITypeSystemService tss = mi.Invoke(null, null) as ITypeSystemService;
				if (tss == null)
					throw new EcoException(HandlesStringRes.sNotATypeSystemProvider(this.ToString()));
				return tss.StaticEcoServices.GetEcoService(serviceType);
			}
		}

		public override bool Active
		{
			get { return Enabled && (EcoSpace != null) && EcoSpace.Active; }
		}

		protected override void ActiveChanged()
		{
			base.ActiveChanged();
			ElementChanged();
		}

		private System.Type m_EcoSpaceType;

		/// <summary>
		/// <para>The type of the ECO Space to which the RootHandle is intended to be a connection.</para>
		///
		/// <para>Set this property to enable the implementation of IStaticContext, even if the EcoSpace property is null.
		/// This enables design-time support for OCL expressions for handles and components connected to the handle.</para>
		/// </summary>
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryTypes")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyEcoSpaceType")]
		[TypeConverter("Borland.Eco.Handles.Design.EcoSpaceTypeTypeConverter, Borland.Eco.Handles.Design")]
		[DefaultValue(null)]
		public System.Type EcoSpaceType
		{
			get { return m_EcoSpaceType; }
			set
			{
				if (value != m_EcoSpaceType)
				{
					m_EcoSpaceType = value;
					StaticContextChanged();
				}
			}
		}
	}
}
